/* main_Z4.c - XBIC example for S32R274 */
/* Description:  Injects error to crossbar */
/* Rev 1.0 Aug 21 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main_Z4.c
* Owner             David Chung
* Version           1.0
* Date              Aug-21-2018
* Classification    General Business Information
* Brief            	XBIC test for DMA master->SRAM slave
*
********************************************************************************
* Detailed Description:
* Tests the XBAR integrity checker. Configures 200 MHz PLL and DMA. The XBIC is
* configured to inject (i.e. simulate) an error to DMA-SRAM communication.  The
* DMA transfer is then activated and UART terminal reports the error detected.
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB+MPC57xxMG
* MCU:             S32R274
* Terminal:        19200, 8N1, None
* Fsys:            200 MHz PLL on 40 MHz external oscillator
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Two options to connect to UART:
*					1) Connect miniUSB port J12 (S32R274RRUEVB) to PC via
*						miniUSB adapter. You must remove jumper J25 (MPC57xxMB) to power
*						off the MPC57xxMB UART port. Both ports map to the same
*						LINFlexD module as J12 (S32R274RRUEVB). This contention
*						causes problems in UART reception unless the
*						motherboard's UART port is powered off. Refer to
*						"Connection_Setup_Option1.jpg".
*					2) Leave jumper J25 (MPC57xxMB) on and connect to PC from the motherboard.
*						Connect the UART port J19 to the PC via an RS232 to USB adapter. MPC57xxMB's
*						UART port signal overpowers any possible contention from the J12 (S32R274EVB).
*						Refer to "Connection_Setup_Option2.jpg".
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-21-2018  David Chung	  	Initial version

*******************************************************************************/
#include "derivative.h" /* include peripheral declarations */
#include "uart.h"
#include <string.h>

/*******************************************************************************
* Constants and macros
*******************************************************************************/
#define 	DRUN_MODE 	0x3
#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

/*******************************************************************************
* External objects
*******************************************************************************/
extern void xcptn_xmpl(void);

/*******************************************************************************
* Global variables
*******************************************************************************/

/*******************************************************************************
* Local functions
*******************************************************************************/
void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

/*******************************************************************************
Function Name : Sys_Init
Engineer      : David Chung
Date          : Aug-21-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Clock settings
Issues        : NONE
*******************************************************************************/
void Sys_Init(void)
{
    //Clk Out
	MCB.CLKOUT_SEL.B.CLKOUT0 = 0b0010; //Select PLL0_PHI_CLK as source of CLKOUT0
	CGM.AC14_DC0.R = 0x80000000;	//Enable divider and divide by (0+1)

    //Configure RunPeripheralConfiguration registers in ME_RUN_PC0
    MC_ME.RUN_PC[0].B.DRUN = 1;             //Enable DRUN mode for all peripherals using ME_RUN_PC[0]

    //Enable external oscilator
    MC_ME.DRUN_MC.B.XOSCON = 1;

    //AUX Clock Selector 3 setup
    CGM.AC3_SC.B.SELCTL = 1;           //connect (8..40MHz) XTAL to the PLL0 input
    MC_ME.DRUN_MC.B.PLL0ON = 1;             //Enable PLL0 for DRUN mode

    // Set PLL0 to 200MHz
//    PLLDIG.PLL0CR.B.CLKCFG = 1;             //Bypass mode PLL0 on
    // RFDPHI1 = 10, RFDPHI = 2, PREDIV = 2, MFD = 14
    PLLDIG.PLL0DV.R = 0x50000000 |0x00020000 |0x00002000 |0x0014 ; //predefined PLL0 divider register

    //AUX Clock Selector 4 setup
    CGM.AC4_SC.B.SELCTL = 0; //Connect PLL0_PHI1_CLK (40 MHz) as source
    MC_ME.DRUN_MC.B.PLL1ON = 1; //Enable PLL1 in DRUN mode

    //Set PLL1 to 160 MHz
    PLLDIG.PLL1DV.R = 0x00020010; //MFD = 0x10, RFD = 2. 40 * 16/(2*2) = 40 * 4 = 160 MHz

    // Set PLL0 as system clock
    MC_ME.DRUN_MC.B.SYSCLK = 0x2;

    //  Enable system clock divider /4 for PBRIDGEx_CLK
    CGM.SC_DC3.R = 0x80030000;

    /* Enable LINFlexD_1 for all modes. */
    MC_ME.RUN_PC[1].R = 0x000000FE; //Enable RUN_PC[1] for all modes
    MC_ME.PCTL[91].B.RUN_CFG = 0x1; //LINFlexD_1: select RUN_PC[1]

    //Mode transition to apply the PLL0 setup and set Normal mode with PLL running
    MC_ME.MCTL.R = 0x30005AF0;              //DRUN Mode & Key
    MC_ME.MCTL.R = 0x3000A50F;              //DRUN Mode & Key

    while(!MC_ME.GS.B.S_PLL0);              //ME_GS Wait for PLL stabilization.
    while(MC_ME.GS.B.S_MTRANS);             //Waiting for end of transaction
    while(MC_ME.GS.B.S_CURRENT_MODE != DRUN_MODE);  // ME_GS Check DRUN mode has successfully been entered
}//Sys_Init

/*******************************************************************************
Function Name : XBIC_error_injection
Engineer      : b21190
Date          : Apr-30-2014
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Inject the error on XBAR for DMA master.
Issues        : NONE
*******************************************************************************/
void XBIC_error_injection (void)
{
  /* Enable master and slave ports for EDC error detect.
   * Crossbar (AXBSn) physical master/slave port assignments
   * are found in SS21.1.1 of the S32R274RM_Rev4
   */
	XBIC_0.MCR.B.SE0 = 0x1; //Enable slave physical port 0 (SRAM controller port 0 for accesses from Core0/CSE/eDMA/Zipwire/ENET/FlexRay)
	XBIC_0.MCR.B.ME4 = 0x1; //Enable AXBS_0 (crossbar 0) master physical port 4 (DMA)

	/* Configure error injection.
	 * XBIC_EIR uses logical master ID on master side,
     * but physical port number on the slave side. Find the
     * logical master ID in Table 24-1 of the reference manual. The
     * logical master ID is used because multiple masters can be
     * connected to the same crossbar port, while each master has
     * a distinct logical master ID.  Thus, using the logical master
     * ID will enable the XBIC to know exactly which master to
     * inject the error into.  For example, eDMA and SIPI are both
     * hooked to AXBS_0 Master Port 4. If the architecture were
     * to use the physical port number, the XBIC would not know whether
     * to inject the error to a SIPI transaction or an eDMA transfer.
     *
     * That being said, you use the physical port number on the slave side, since
     * only one master can control a slave during any given transaction.
     * Using the slave's physical port number is therefore sufficent even
     * if a slave port is split.
     *
	 * SLV = 0 (Error injection enabled for physical slave port 0 (SRAM controller port 2))
	 * MST = 2 (Error injection is enabled for logical master 2 (DMA))
	 * SYN = xF. This valued is XOR'd with the calculated syndrome.
	 */
	XBIC_0.EIR.R = 0x0000020F;

  /* Error Injection Enable */
  XBIC_0.EIR.B.EIE = 0x1;
}//XBIC_error_injection

/*******************************************************************************
Function Name : AIPS_0_Init
Engineer      : David Chung
Date          : Aug-21-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Setup access right for Masters and Peripherals on AIPS
Issues        : NONE
*******************************************************************************/
void AIPS_0_Init(void)
{
    // All Master Can go through AIPS_0 and all peripheral have no protection

    // Master Privilege Register A
    AIPS_0.MPRA.R      = 0x77777777;
    /* Peripheral Access Control Register.
     * Unused PACRn/OPACRn registers are
     * reserved. Check Columns B-C of the
     * "Peripheral Map" in S32R274_Memory_Map_Rev15.xlsx
     * to see which registers are implemented.
     */
    AIPS_0.PACR[0].R    = 0x0;
    AIPS_0.PACR[1].R    = 0x0;
    AIPS_0.PACR[2].R    = 0x0;
    AIPS_0.PACR[3].R    = 0x0;
//    AIPS_0.PACR[4].R    = 0x0; //Writing to this led to machine check error despite the memory table
    AIPS_0.PACR[5].R    = 0x0;
    AIPS_0.PACR[6].R    = 0x0;
    AIPS_0.PACR[7].R    = 0x0;
//    AIPS_0.PACR[8].R    = 0x0; //Writing to this led to machine check error despite the memory table
    AIPS_0.PACR[9].R    = 0x0;
    AIPS_0.PACR[10].R    = 0x0;
//    AIPS_0.PACR[11].R    = 0x0;
//    AIPS_0.PACR[12].R    = 0x0; //Writing to this led to machine check error despite the memory table
    AIPS_0.PACR[13].R    = 0x0;
//    AIPS_0.PACR[14].R    = 0x0;
    AIPS_0.PACR[15].R    = 0x0;
    AIPS_0.PACR[16].R    = 0x0;
    AIPS_0.PACR[17].R    = 0x0;
//    AIPS_0.PACR[18].R    = 0x0;
//    AIPS_0.PACR[19].R    = 0x0;
    AIPS_0.PACR[20].R    = 0x0;
    AIPS_0.PACR[21].R    = 0x0;
    AIPS_0.PACR[22].R    = 0x0;
//    AIPS_0.PACR[23].R    = 0x0;
//    AIPS_0.PACR[24].R    = 0x0;
//    AIPS_0.PACR[25].R    = 0x0;
    AIPS_0.PACR[26].R    = 0x0;
    AIPS_0.PACR[27].R    = 0x0;
    AIPS_0.PACR[28].R    = 0x0;
    AIPS_0.PACR[29].R    = 0x0;
    AIPS_0.PACR[30].R    = 0x0;
    AIPS_0.PACR[31].R    = 0x0;
//    AIPS_0.PACR[32].R    = 0x0;
//    AIPS_0.PACR[33].R    = 0x0;
//    AIPS_0.PACR[34].R    = 0x0;
//    AIPS_0.PACR[35].R    = 0x0;
//    AIPS_0.PACR[36].R    = 0x0;
//    AIPS_0.PACR[37].R    = 0x0;
//    AIPS_0.PACR[38].R    = 0x0;
//    AIPS_0.PACR[39].R    = 0x0;
    AIPS_0.PACR[40].R    = 0x0;
//    AIPS_0.PACR[41].R    = 0x0;
//    AIPS_0.PACR[42].R    = 0x0;
//    AIPS_0.PACR[43].R    = 0x0;
    AIPS_0.PACR[44].R    = 0x0;
//    AIPS_0.PACR[45].R    = 0x0;
//    AIPS_0.PACR[46].R    = 0x0;
//    AIPS_0.PACR[47].R    = 0x0;
//    AIPS_0.PACR[48].R    = 0x0;
//    AIPS_0.PACR[49].R    = 0x0;
//    AIPS_0.PACR[50].R    = 0x0;
//    AIPS_0.PACR[51].R    = 0x0;
//    AIPS_0.PACR[52].R    = 0x0;
//    AIPS_0.PACR[53].R    = 0x0;
//    AIPS_0.PACR[54].R    = 0x0;
//    AIPS_0.PACR[55].R    = 0x0;
//    AIPS_0.PACR[56].R    = 0x0;
//    AIPS_0.PACR[57].R    = 0x0;
//    AIPS_0.PACR[58].R    = 0x0;
//    AIPS_0.PACR[59].R    = 0x0;
//    AIPS_0.PACR[60].R    = 0x0;
//    AIPS_0.PACR[61].R    = 0x0;
//    AIPS_0.PACR[62].R    = 0x0;
//    AIPS_0.PACR[63].R    = 0x0;

    // Off-Platform Peripheral Access Control Register
    AIPS_0.OPACR[0].R	= 0x0;
    AIPS_0.OPACR[1].R	= 0x0;
    AIPS_0.OPACR[2].R	= 0x0;
//    AIPS_0.OPACR[3].R	= 0x0;
//    AIPS_0.OPACR[4].R	= 0x0;
//    AIPS_0.OPACR[5].R	= 0x0;
//    AIPS_0.OPACR[6].R	= 0x0;
    AIPS_0.OPACR[7].R	= 0x0;
//    AIPS_0.OPACR[8].R	= 0x0;
    AIPS_0.OPACR[9].R	= 0x0;
//    AIPS_0.OPACR[10].R	= 0x0;
    AIPS_0.OPACR[11].R	= 0x0;
//    AIPS_0.OPACR[12].R	= 0x0;
//    AIPS_0.OPACR[13].R	= 0x0;
//    AIPS_0.OPACR[14].R	= 0x0;
    AIPS_0.OPACR[15].R	= 0x0;
//    AIPS_0.OPACR[16].R	= 0x0;
    AIPS_0.OPACR[17].R	= 0x0;
//    AIPS_0.OPACR[18].R	= 0x0;
    AIPS_0.OPACR[19].R	= 0x0;
//    AIPS_0.OPACR[20].R	= 0x0;
    AIPS_0.OPACR[21].R	= 0x0;
//    AIPS_0.OPACR[22].R	= 0x0;
    AIPS_0.OPACR[23].R	= 0x0;
//    AIPS_0.OPACR[24].R	= 0x0;
    AIPS_0.OPACR[25].R	= 0x0;
//    AIPS_0.OPACR[26].R	= 0x0;
//    AIPS_0.OPACR[27].R	= 0x0;
//    AIPS_0.OPACR[28].R	= 0x0;
//    AIPS_0.OPACR[29].R	= 0x0;
    AIPS_0.OPACR[30].R	= 0x0;
    AIPS_0.OPACR[31].R	= 0x0;
//    AIPS_0.OPACR[32].R	= 0x0;
//    AIPS_0.OPACR[33].R	= 0x0;
//    AIPS_0.OPACR[34].R	= 0x0;
//    AIPS_0.OPACR[35].R	= 0x0;
    AIPS_0.OPACR[36].R	= 0x0;
//    AIPS_0.OPACR[37].R	= 0x0;
    AIPS_0.OPACR[38].R	= 0x0;
//    AIPS_0.OPACR[39].R	= 0x0;
//    AIPS_0.OPACR[40].R	= 0x0;
//    AIPS_0.OPACR[41].R	= 0x0;
//    AIPS_0.OPACR[42].R	= 0x0;
    AIPS_0.OPACR[43].R	= 0x0;
//    AIPS_0.OPACR[44].R	= 0x0;
    AIPS_0.OPACR[45].R	= 0x0;
    AIPS_0.OPACR[46].R	= 0x0;
//    AIPS_0.OPACR[47].R	= 0x0;
//    AIPS_0.OPACR[48].R	= 0x0;
//    AIPS_0.OPACR[49].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[50].R	= 0x0;
//    AIPS_0.OPACR[51].R	= 0x0;
//    AIPS_0.OPACR[52].R	= 0x0;
//    AIPS_0.OPACR[53].R	= 0x0;
//    AIPS_0.OPACR[54].R	= 0x0;
//    AIPS_0.OPACR[55].R	= 0x0;
//    AIPS_0.OPACR[56].R	= 0x0;
//    AIPS_0.OPACR[57].R	= 0x0;
//    AIPS_0.OPACR[58].R	= 0x0;
//    AIPS_0.OPACR[59].R	= 0x0;
//    AIPS_0.OPACR[60].R	= 0x0;
//    AIPS_0.OPACR[61].R	= 0x0;
//    AIPS_0.OPACR[62].R	= 0x0;
//    AIPS_0.OPACR[63].R	= 0x0;
//    AIPS_0.OPACR[64].R	= 0x0;
//    AIPS_0.OPACR[65].R	= 0x0;
//    AIPS_0.OPACR[66].R	= 0x0;
//    AIPS_0.OPACR[67].R	= 0x0;
//    AIPS_0.OPACR[68].R	= 0x0;
//    AIPS_0.OPACR[69].R	= 0x0;
//    AIPS_0.OPACR[70].R	= 0x0;
//    AIPS_0.OPACR[71].R	= 0x0;
//    AIPS_0.OPACR[72].R	= 0x0;
//    AIPS_0.OPACR[72].R	= 0x0;
//    AIPS_0.OPACR[73].R	= 0x0;
//    AIPS_0.OPACR[74].R	= 0x0;
//    AIPS_0.OPACR[75].R	= 0x0;
//    AIPS_0.OPACR[76].R	= 0x0;
//    AIPS_0.OPACR[77].R	= 0x0;
//    AIPS_0.OPACR[78].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[79].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[80].R	= 0x0;
//    AIPS_0.OPACR[81].R	= 0x0;
//    AIPS_0.OPACR[82].R	= 0x0;
//    AIPS_0.OPACR[83].R	= 0x0;
//    AIPS_0.OPACR[84].R	= 0x0;
//    AIPS_0.OPACR[85].R	= 0x0;
//    AIPS_0.OPACR[86].R	= 0x0;
//    AIPS_0.OPACR[87].R	= 0x0;
//    AIPS_0.OPACR[88].R	= 0x0;
//    AIPS_0.OPACR[89].R	= 0x0;
//    AIPS_0.OPACR[90].R	= 0x0;
//    AIPS_0.OPACR[91].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[92].R	= 0x0;
//    AIPS_0.OPACR[93].R	= 0x0;
//    AIPS_0.OPACR[94].R	= 0x0;
//    AIPS_0.OPACR[95].R	= 0x0;
//    AIPS_0.OPACR[96].R	= 0x0;
//    AIPS_0.OPACR[97].R	= 0x0;
//    AIPS_0.OPACR[98].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[99].R	= 0x0;
//    AIPS_0.OPACR[100].R	= 0x0;
//    AIPS_0.OPACR[101].R	= 0x0;
//    AIPS_0.OPACR[102].R	= 0x0;
//    AIPS_0.OPACR[103].R	= 0x0;
//    AIPS_0.OPACR[104].R	= 0x0;
//    AIPS_0.OPACR[105].R	= 0x0;
//    AIPS_0.OPACR[106].R	= 0x0;
//    AIPS_0.OPACR[107].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[108].R	= 0x0;
//    AIPS_0.OPACR[109].R	= 0x0;
//    AIPS_0.OPACR[110].R	= 0x0;
//    AIPS_0.OPACR[111].R	= 0x0;
//    AIPS_0.OPACR[112].R	= 0x0;
//    AIPS_0.OPACR[113].R	= 0x0;
//    AIPS_0.OPACR[114].R	= 0x0;
//    AIPS_0.OPACR[115].R	= 0x0;
//    AIPS_0.OPACR[116].R	= 0x0;
//    AIPS_0.OPACR[117].R	= 0x0;
//    AIPS_0.OPACR[118].R	= 0x0;
//    AIPS_0.OPACR[119].R	= 0x0;
//    AIPS_0.OPACR[120].R	= 0x0;
//    AIPS_0.OPACR[121].R	= 0x0;
//    AIPS_0.OPACR[122].R	= 0x0;
//    AIPS_0.OPACR[123].R	= 0x0;
//    AIPS_0.OPACR[124].R	= 0x0;
//    AIPS_0.OPACR[125].R	= 0x0;
//    AIPS_0.OPACR[126].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[127].R	= 0x0;
//    AIPS_0.OPACR[128].R	= 0x0;
//    AIPS_0.OPACR[129].R	= 0x0;
//    AIPS_0.OPACR[130].R	= 0x0;
//    AIPS_0.OPACR[131].R	= 0x0;
//    AIPS_0.OPACR[132].R	= 0x0;
//    AIPS_0.OPACR[133].R	= 0x0;
//    AIPS_0.OPACR[134].R	= 0x0;
//    AIPS_0.OPACR[135].R	= 0x0;
//    AIPS_0.OPACR[136].R	= 0x0;
//    AIPS_0.OPACR[137].R	= 0x0;
//    AIPS_0.OPACR[138].R	= 0x0;
//    AIPS_0.OPACR[139].R	= 0x0;
//    AIPS_0.OPACR[140].R	= 0x0;
//    AIPS_0.OPACR[141].R	= 0x0;
//    AIPS_0.OPACR[142].R	= 0x0;
//    AIPS_0.OPACR[143].R	= 0x0;
//    AIPS_0.OPACR[144].R	= 0x0;
//    AIPS_0.OPACR[145].R	= 0x0;
//    AIPS_0.OPACR[146].R	= 0x0;
//    AIPS_0.OPACR[147].R	= 0x0;
//    AIPS_0.OPACR[148].R	= 0x0;
//    AIPS_0.OPACR[149].R	= 0x0;
//    AIPS_0.OPACR[150].R	= 0x0;
//    AIPS_0.OPACR[151].R	= 0x0;
//    AIPS_0.OPACR[152].R	= 0x0;
//    AIPS_0.OPACR[153].R	= 0x0;
//    AIPS_0.OPACR[154].R	= 0x0;
//    AIPS_0.OPACR[155].R	= 0x0;
//    AIPS_0.OPACR[156].R	= 0x0;
//    AIPS_0.OPACR[157].R	= 0x0;
//    AIPS_0.OPACR[158].R	= 0x0;
//    AIPS_0.OPACR[159].R	= 0x0;
//    AIPS_0.OPACR[160].R	= 0x0;
//    AIPS_0.OPACR[161].R	= 0x0;
//    AIPS_0.OPACR[162].R	= 0x0;
//    AIPS_0.OPACR[163].R	= 0x0;
//    AIPS_0.OPACR[164].R	= 0x0;
//    AIPS_0.OPACR[165].R	= 0x0;
//    AIPS_0.OPACR[166].R	= 0x0;
//    AIPS_0.OPACR[167].R	= 0x0;
//    AIPS_0.OPACR[168].R	= 0x0;
//    AIPS_0.OPACR[169].R	= 0x0;
//    AIPS_0.OPACR[170].R	= 0x0;
//    AIPS_0.OPACR[171].R	= 0x0;
//    AIPS_0.OPACR[172].R	= 0x0;
//    AIPS_0.OPACR[173].R	= 0x0;
//    AIPS_0.OPACR[174].R	= 0x0;
//    AIPS_0.OPACR[175].R	= 0x0;
//    AIPS_0.OPACR[176].R	= 0x0;
//    AIPS_0.OPACR[177].R	= 0x0;
//    AIPS_0.OPACR[178].R	= 0x0;
//    AIPS_0.OPACR[179].R	= 0x0;
//    AIPS_0.OPACR[180].R	= 0x0;
//    AIPS_0.OPACR[181].R	= 0x0;
//    AIPS_0.OPACR[182].R	= 0x0;
//    AIPS_0.OPACR[183].R	= 0x0;
//    AIPS_0.OPACR[184].R	= 0x0;
//    AIPS_0.OPACR[185].R	= 0x0;
//    AIPS_0.OPACR[186].R	= 0x0;
//    AIPS_0.OPACR[187].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[188].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[189].R	= 0x0;
//    AIPS_0.OPACR[190].R	= 0x0;
//    AIPS_0.OPACR[191].R	= 0x0;
//    AIPS_0.OPACR[192].R	= 0x0;
//    AIPS_0.OPACR[193].R	= 0x0;
//    AIPS_0.OPACR[194].R	= 0x0;
//    AIPS_0.OPACR[195].R	= 0x0;
//    AIPS_0.OPACR[196].R	= 0x0;
//    AIPS_0.OPACR[197].R	= 0x0;
//    AIPS_0.OPACR[198].R	= 0x0;
//    AIPS_0.OPACR[199].R	= 0x0;
//    AIPS_0.OPACR[200].R	= 0x0;
//    AIPS_0.OPACR[201].R	= 0x0;
//    AIPS_0.OPACR[202].R	= 0x0;
//    AIPS_0.OPACR[203].R	= 0x0;
//    AIPS_0.OPACR[204].R	= 0x0;
//    AIPS_0.OPACR[205].R	= 0x0;
//    AIPS_0.OPACR[206].R	= 0x0;
//    AIPS_0.OPACR[207].R	= 0x0;
//    AIPS_0.OPACR[208].R	= 0x0;
//    AIPS_0.OPACR[209].R	= 0x0;
//    AIPS_0.OPACR[210].R	= 0x0;
//    AIPS_0.OPACR[211].R	= 0x0;
//    AIPS_0.OPACR[212].R	= 0x0;
//    AIPS_0.OPACR[213].R	= 0x0;
//    AIPS_0.OPACR[214].R	= 0x0;
//    AIPS_0.OPACR[215].R	= 0x0;
//    AIPS_0.OPACR[216].R	= 0x0;
//    AIPS_0.OPACR[217].R	= 0x0;
//    AIPS_0.OPACR[218].R	= 0x0;
//    AIPS_0.OPACR[219].R	= 0x0;
//    AIPS_0.OPACR[220].R	= 0x0;
//    AIPS_0.OPACR[221].R	= 0x0;
//    AIPS_0.OPACR[222].R	= 0x0;
//    AIPS_0.OPACR[223].R	= 0x0;
//    AIPS_0.OPACR[224].R	= 0x0;
//    AIPS_0.OPACR[225].R	= 0x0;
//    AIPS_0.OPACR[226].R	= 0x0;
//    AIPS_0.OPACR[227].R	= 0x0;
//    AIPS_0.OPACR[228].R	= 0x0;
//    AIPS_0.OPACR[229].R	= 0x0;
//    AIPS_0.OPACR[230].R	= 0x0;
//    AIPS_0.OPACR[231].R	= 0x0;
//    AIPS_0.OPACR[232].R	= 0x0;
//    AIPS_0.OPACR[233].R	= 0x0;
//    AIPS_0.OPACR[234].R	= 0x0;
//    AIPS_0.OPACR[235].R	= 0x0;
//    AIPS_0.OPACR[236].R	= 0x0;
//    AIPS_0.OPACR[237].R	= 0x0;
//    AIPS_0.OPACR[238].R	= 0x0;
//    AIPS_0.OPACR[239].R	= 0x0;
//    AIPS_0.OPACR[240].R	= 0x0;
//    AIPS_0.OPACR[241].R	= 0x0;
//    AIPS_0.OPACR[242].R	= 0x0;
//    AIPS_0.OPACR[243].R	= 0x0;
//    AIPS_0.OPACR[244].R	= 0x0;
//    AIPS_0.OPACR[245].R	= 0x0;
//    AIPS_0.OPACR[246].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[247].R	= 0x0;
//    AIPS_0.OPACR[248].R	= 0x0;
//    AIPS_0.OPACR[249].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[250].R	= 0x0;
//    AIPS_0.OPACR[251].R	= 0x0;
//    AIPS_0.OPACR[252].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[253].R	= 0x0; //Writing to this led to machine check error despite the memory table
//    AIPS_0.OPACR[254].R	= 0x0;
//    AIPS_0.OPACR[255].R	= 0x0; //Writing to this led to machine check error despite the memory table
}//AIPS_0_Init

/*******************************************************************************
Function Name : DMA_0_Init
Engineer      : David Chung
Date          : Aug-21-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : DMA initialization funtion initialize TCD for simple transfer
                from constant "source" stored in internal flash to start of SRAM
				to demonstrate correct configuration. Used is one major loop.
				DMA is started by SW.
Issues        : NONE
*******************************************************************************/
void DMA_0_Init(void)
{
    // Clear the ENBL and TRIG bits of the DMA channel
    DMAMUX_0.CHCFG[0].B.ENBL = 0;
    DMAMUX_0.CHCFG[0].B.TRIG = 0;

    // DMA config
    DMA.ERQ.B.ERQ0 = 0;    //The DMA request signal for channel 0 is disabled.
    //TCD config for channel[0]

    // TCD[0] Word 0 config
    DMA.TCD[0].SADDR.R = 0x40000000;  	//Source Address

    // TCD[0] Word 1 config SMOD(0) | SSIZE(32-bit) | DMOD(0) | DSIZE(32-bit)
    DMA.TCD[0].ATTR.R = 0x0|0x200|0x0|0x2;	//Source transfer size 32-bit, no Address Modulo used

    // TCD[0] Word 2 config NBYTES - Minor Byte Transfer Count
    // Number of bytes to be transferred in each service request of the channel
    DMA.TCD[0].NBYTES.MLNO.R = 0x4;

    // TCD[0] Word 3 config SLAST - TCD Last Source Address Adjustment
    DMA.TCD[0].SLAST.R = 0;

    // TCD[0] Word 4 config DADDR - TCD Destination Address
    DMA.TCD[0].DADDR.R = 0x40001000;	 //Destination Address

    // TCD[0] Word 5 config CITER - TCD Current Minor Loop Link, Major Loop Count
    // ELINK | CITER
    DMA.TCD[0].CITER.ELINKNO.R = 0x0 | 0x1;  //Destination Address

    // TCD[0] Word 6 config DLAST_SGA - TCD Last Destination Address Adjustment/Scatter Gather Address
    DMA.TCD[0].DLASTSGA.R = 0x4;	 // Destination last address adjustment

    // TCD[0] Word 7 config BITER - TCD Beginning Minor Loop Link, Major Loop Count
    // ELINK | BITER
    DMA.TCD[0].BITER.ELINKNO.R = 0x0 | 0x1;	 // Destination last address adjustment

    DMA.SERQ.R = 0x0; //Enable Channel 0
}//DMA_0_Init

/*******************************************************************************
Function Name : HexToASCII_Word
Engineer      : b48683
Date          : Jan-30-2017
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Turns hex word to ASCII and stores in buffer
Issues        : NONE
*******************************************************************************/
void HexToASCII_Word(const unsigned int input,uint8_t* buffer){
	uint8_t i = 0;
	unsigned int number = input;
	unsigned int temp = 0;

	for(i=0;i<8;i++){
		temp = number & 0x0000000F; //Mask least significant nibble of word
		/* Convert to ASCII. Store in buffer backwards because LSB gets stored first */
		if(temp > 9){
			*(buffer+8-(i+1)) = temp + '0' + 7; //'A' does not come immediately after '9' in ASCII table. Additional offset
		}else{
			*(buffer+8-(i+1)) = temp + '0'; //Otherwise store '0' to '9'
		}

		number = number >> 4; //Get next nibble
	}
}

/*******************************************************************************
* Global functions
*******************************************************************************/
int main(void)
{
	char dummychar = 0;
	uint8_t	intro[] = {"\n\rWelcome to the S32R274 XBIC code example! "
			"XBIC is the crossbar integrity checker, which checks the consistency "
			"between signals on the master side and what comes out on the slave side,"
			"and vice versa.\n\r"};
	uint8_t intro2[] = {"\n\rDMA is configured to transfer PRAM to PRAM."
			" XBIC is configured to inject an error for DMA-PRAM communication. DMA is master 4 (M4) and "
			"SRAM controller port 0 (for Core0, eDMA, etc. accesses) is slave 0 (S0).\n\r"};
	uint8_t Continue_Message[] = {"\n\rPress any key to continue...\n\r"};
	uint8_t Message0[] = {"\n\rDMA transfer of started.\n\r"};
	uint8_t Message1[] = {"\n\rAn error was detected. Syndrome 0x"};
	uint8_t Message2[] = {"\n\rThe master associated with the error is M"};
	uint8_t Message3[] = {"\n\rThe slave associated with the error is S"};
	uint8_t Message4[] = {"\n\rThe error was encountered at address 0x"};
	uint8_t Message5[] = {"\n\rNo crossbar error detected.\n\r"};
	uint8_t newline[] = {".\n\r"};

	uint8_t PrintBuffer[8] = {0};
	uint8_t master = 0;
	uint8_t slave = 0;
	uint8_t syndrome = 0;
	uint32_t address = 0;

	xcptn_xmpl();
	Sys_Init();
	AIPS_0_Init();
    DMA_0_Init();
    LINFlexD_1_Init(); //Initialize LINFlexD_1 for printing UART messages to computer terminal
    TransmitData((const char*)intro,strlen((const char*)intro));
    TransmitData((const char*)Continue_Message,strlen((const char*)Continue_Message));
    ReceiveData((char*)&dummychar);
    TransmitData((const char*)intro2,strlen((const char*)intro2));

    XBIC_error_injection();

    while(1){
    	TransmitData((const char*)Message0,strlen((const char*)Message0));

    	/* Start DMA */
    	DMA.SSRT.R = 0; 	// Start DMA transfer ROM to RAM

    	syndrome = XBIC_0.ESR.R & 0x000000FF;
    	master = ((XBIC_0.ESR.R & 0x00000F00) >> 8) + '0'; //Turn into ASCII
    	slave = ((XBIC_0.ESR.R & 0x00007000) >> 12) + '0'; //Turn into ASCII
    	address = XBIC_0.EAR.R;

    	/* Print error injection success/failure */
    	if(XBIC_0.ESR.B.VLD){
        	/* Talk about syndrome if error detected. */
        	HexToASCII_Word(syndrome,PrintBuffer);
        	TransmitData((const char*)Message1,strlen((const char*)Message1));
        	TransmitData((const char*)PrintBuffer,8);
        	TransmitData((const char*)newline,strlen((const char*)newline));

        	/* Print message about master */
        	TransmitData((const char*)Message2,strlen((const char*)Message2));
        	TransmitData((const char*)&master,1);
        	TransmitData((const char*)newline,strlen((const char*)newline));

        	/* Print message about slave */
        	TransmitData((const char*)Message3,strlen((const char*)Message3));
        	TransmitData((const char*)&slave,1);
        	TransmitData((const char*)newline,strlen((const char*)newline));

        	/* Print message about error address */
        	HexToASCII_Word(address,PrintBuffer);
        	TransmitData((const char*)Message4,strlen((const char*)Message4));
        	TransmitData((const char*)PrintBuffer,8);
        	TransmitData((const char*)newline,strlen((const char*)newline));

        	/* Await user response before repeat */
        	TransmitData((const char*)Continue_Message,strlen((const char*)Continue_Message));
    	}else{
    		/* Tell user that no error has been detected */
    		TransmitData((const char*)Message5,strlen((const char*)Message5));
    	}

    	ReceiveData((char*)&dummychar);
    }
    return 0;
}


